import { LastFmService } from './lastfmService'
import { JamendoService, type JamendoTrack } from './jamendoService'

export interface Song {
  id: string
  title: string
  artist: string
  duration: number
  url: string
  imageUrl: string
  tags: string[]
}

export class RecommendationService {
  private lastfmService: LastFmService
  private jamendoService: JamendoService

  constructor() {
    this.lastfmService = new LastFmService()
    this.jamendoService = new JamendoService()
  }

  async getRecommendations(
    emotion: string,
    intensity: number,
    userTags: string[],
    limit: number = 5
  ): Promise<Song[]> {
    try {
      // 使用 Jamendo API 搜索音乐
      const { query, tags } = this.jamendoService.mapEmotionToSearchTerms(emotion, intensity)
      let tracks: JamendoTrack[] = []

      // 首先尝试使用情绪和标签组合搜索
      tracks = await this.jamendoService.searchTracks(query, [...tags, ...userTags], limit)

      // 如果没有找到结果，只使用情绪标签
      if (tracks.length === 0) {
        console.log('No results with user tags, trying emotion tags only...')
        tracks = await this.jamendoService.searchTracks(query, tags, limit)
      }

      // 如果还是没有结果，使用备用标签
      if (tracks.length === 0) {
        console.log('No results with emotion tags, trying fallback tags...')
        const fallbackTags = this.getFallbackTags(emotion)
        tracks = await this.jamendoService.searchTracks('', fallbackTags, limit)
      }

      // 如果仍然没有结果，返回一些流行音乐
      if (tracks.length === 0) {
        console.log('No results with fallback tags, getting popular music...')
        tracks = await this.jamendoService.searchTracks('', ['pop', 'rock'], limit)
      }

      return tracks.map(track => ({
        id: track.id,
        title: track.name,
        artist: track.artist_name,
        duration: track.duration,
        url: track.audio,
        imageUrl: track.image,
        tags: track.tags
      }))
    } catch (error) {
      console.error('Error getting recommendations:', error)
      throw error
    }
  }

  private getFallbackTags(emotion: string): string[] {
    const fallbackMap: Record<string, string[]> = {
      'Happy': ['pop', 'dance', 'electronic'],
      'Sad': ['classical', 'ambient', 'piano'],
      'Angry': ['rock', 'metal', 'electronic'],
      'Relaxed': ['classical', 'jazz', 'ambient'],
      'Excited': ['electronic', 'dance', 'rock']
    }
    return fallbackMap[emotion] || ['pop', 'rock']
  }
}
