import { ref } from 'vue'
import { useToast } from './useToast'
import { posts } from '../utils/api'

export function usePosts() {
  const { showToast } = useToast()
  const postsList = ref([])
  const currentPost = ref(null)
  const loading = ref(false)
  const error = ref(null)

  const fetchPosts = async (params = {}) => {
    loading.value = true
    error.value = null
    try {
      const { data } = await posts.list(params)
      return {
        items: data.items || [],
        total: data.total || 0
      }
    } catch (err) {
      error.value = err.response?.data?.message || '获取帖子列表失败'
      showToast(error.value, 'error')
      throw err
    } finally {
      loading.value = false
    }
  }

  const getPost = async (id) => {
    loading.value = true
    error.value = null
    try {
      const { data } = await posts.getById(id)
      currentPost.value = data
      return data
    } catch (err) {
      error.value = err.response?.data?.message || '获取帖子详情失败'
      showToast(error.value, 'error')
      throw err
    } finally {
      loading.value = false
    }
  }

  const createPost = async (postData) => {
    error.value = null
    loading.value = true
    try {
      const response = await posts.create(postData)
      showToast('发布成功', 'success')
      // 返回响应数据中的post对象
      return response.data[1]  // [201, post]
    } catch (err) {
      error.value = err.response?.data?.message || '创建帖子失败'
      showToast(error.value, 'error')
      throw err
    } finally {
      loading.value = false
    }
  }

  const updatePost = async (id, postData) => {
    error.value = null
    loading.value = true
    try {
      const response = await posts.update(id, postData)
      showToast('更新成功', 'success')
      // 返回响应数据中的post对象
      return response.data[1]  // [200, post]
    } catch (err) {
      error.value = err.response?.data?.message || '更新帖子失败'
      showToast(error.value, 'error')
      throw err
    } finally {
      loading.value = false
    }
  }

  const deletePost = async (id) => {
    loading.value = true
    error.value = null
    try {
      await posts.delete(id)
      showToast('删除成功', 'success')
    } catch (err) {
      error.value = err.response?.data?.message || '删除帖子失败'
      showToast(error.value, 'error')
      throw err
    } finally {
      loading.value = false
    }
  }

  const likePost = async (id) => {
    try {
      await posts.like(id)
      if (currentPost.value && currentPost.value.id === id) {
        currentPost.value.likes_count++
        currentPost.value.is_liked = true
      }
    } catch (err) {
      showToast(err.response?.data?.message || '点赞失败', 'error')
      throw err
    }
  }

  const unlikePost = async (id) => {
    try {
      await posts.unlike(id)
      if (currentPost.value && currentPost.value.id === id) {
        currentPost.value.likes_count--
        currentPost.value.is_liked = false
      }
    } catch (err) {
      showToast(err.response?.data?.message || '取消点赞失败', 'error')
      throw err
    }
  }

  return {
    postsList,
    currentPost,
    loading,
    error,
    fetchPosts,
    getPost,
    createPost,
    updatePost,
    deletePost,
    likePost,
    unlikePost
  }
}
